package com.Rokato.jni_demo;

import androidx.appcompat.app.AppCompatActivity;

import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.widget.Button;
import android.widget.EditText;
import android.widget.Spinner;

import com.Rokato.jni.Rokato;
import com.Rokato.jni.rf.CosErr;
import com.Rokato.jni.rf.CpuCard;
import com.Rokato.jni.rf.IRfReader;
import com.Rokato.jni.rf.LoopReqCard;
import com.Rokato.jni.rf.NfcReader;
import com.Rokato.jni.rf.RfReader;
import com.Rokato.jni.utils.Convert;

import java.util.ArrayList;
import java.util.List;

public class CpuToolActivity extends AppCompatActivity {
    private static final String TAG = "Rokato_CpuTool";
    private Spinner devTypeSpinner;
    private EditText serialAddrEdit;
    private EditText logEdit;
    private Button cpuResetBtn;
    private boolean cpuSession;
    private EditText apduDataEdit;
    private RfReader rf;
    private NfcReader nfc;
    private LoopReqCard loop;
    private Handler mainHandler;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_cpu_tool);

        devTypeSpinner = findViewById(R.id.devTypeSel_spinner);
        serialAddrEdit = findViewById(R.id.serialAddr_edit);
        apduDataEdit = findViewById(R.id.apduData_edit);

        logEdit = findViewById(R.id.log_edit);

        Button clearLogBtn = findViewById(R.id.clearLog_button);
        clearLogBtn.setOnClickListener(v -> logEdit.setText(""));
        Log.d(TAG, "onCreate: 构造初始化");
        rf = new RfReader(getApplicationContext(), RfReader.DevType_AUTO, RfReader.RfType_ISO14443A);
        nfc = new NfcReader(getApplicationContext());
        loop = new LoopReqCard(100);
        mainHandler = new Handler(Looper.getMainLooper());

        addLog("JNI详细信息", Rokato.version() + " " + Rokato.build() + "\n");

        Button cfgSerialBtn = findViewById(R.id.cfgSerial_button);
        cfgSerialBtn.setOnClickListener(view -> {
            if (rf.serialCfg((byte) devTypeSpinner.getSelectedItemPosition(), String.valueOf(serialAddrEdit.getText())))
                addLog(devTypeSpinner.getSelectedItem().toString() + " 配置串口完成", "");
            else
                addLog(devTypeSpinner.getSelectedItem().toString() + " 不支持配置串口", "");
        });

        nfc.bind(this, tag -> {
            nfc.setNewTag(tag);
            loop.activeLoop();
        });

        Button openDevBtn = findViewById(R.id.openDev_button);
        openDevBtn.setOnClickListener(v -> {
            if (loop.start()) {
                addLog("读卡器工作中,请刷卡！", "");
                return;
            }
            final int pos = devTypeSpinner.getSelectedItemPosition();
            List<IRfReader> devs = new ArrayList<>();
            if (rf.setDevType((byte) pos)) devs.add(rf);
            if (pos == 0 || pos == 7) devs.add(nfc);
            if (loop.start(false, devs, (cardNo, type, sak) -> {
                addLog("发现卡片:" + Convert.getCardNoStr(cardNo), "类型:" + RfReader.typeStr(type) + "  SAK:" + Convert.bytesToHexStr(sak) + "\n");
                rf.beep((byte) 15);
                if (sak[0] > 0 && (byte) (sak[0] & 0x20) != 0x20) {
                    addLog("此卡可能不是CPU卡,无法复位激活COS会话!", "");
                    return true;
                }
                return switchSession();
            })) {
                if (loop.getRfReader() instanceof RfReader)
                    addLog("打开RF读卡器设备完成", "固件版本:" + rf.getVer() + "\n");
                else if (loop.getRfReader() instanceof NfcReader)
                    addLog("打开NFC读卡器监听完成", "");
                else
                    addLog("未知读卡器启动完成", "");
                return;
            }
            addLog("打开读卡器设备失败！", "请检查读卡器设备或权限是否正常后再试\n");
        });

        Button closeDevBtn = findViewById(R.id.closeDev_button);
        closeDevBtn.setOnClickListener(view -> {
            loop.stop();
            cpuSession = false;
            cpuResetBtn.setText(getString(R.string.cpuReset));
            addLog("关闭读卡器设备完成", "");
        });

        cpuSession = false;
        cpuResetBtn = findViewById(R.id.cpuReset_button);
        cpuResetBtn.setOnClickListener(v -> {
            if (loop.activeReader(this::switchSession)) return;
            addLog("请先打开读卡器设备后操作！", "");
        });

        Button sendApduBtn = findViewById(R.id.sendApdu_button);
        sendApduBtn.setOnClickListener(v -> {
            if (loop.activeReader(this::sendApdu)) return;
            addLog("请先打开读卡器设备后操作！", "");
        });
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        loop.stop();
        rf.free();
        rf = null;
        nfc = null;
        Log.d(TAG, "onDestroy: 释放完成");
    }

    public void addLog(String tag, String msg) {
        if (Looper.myLooper() == Looper.getMainLooper()) {
            MainActivity.addLog(logEdit, tag, msg);
        } else {
            mainHandler.post(() -> MainActivity.addLog(logEdit, tag, msg));
        }
    }

    public boolean switchSession() {
        final CpuCard cpu = loop.getCpuCard();
        if (cpuSession) {
            cpu.halt();
            cpuSession = false;
            mainHandler.post(() -> cpuResetBtn.setText(getString(R.string.cpuReset)));
            addLog("COS会话主动终止完成", "");
            if (!loop.start()) addLog("工作线程解除暂停异常！", "");
            return true;
        }
        byte[] type = new byte[2];
        cpuSession = cpu.reset(false, type);
        if (!cpuSession) {
            addLog("卡片复位激活失败！", "");
            return false;
        }
        loop.pause();
        mainHandler.post(() -> cpuResetBtn.setText(getString(R.string.cpuHalt)));
        addLog(cpu.atsStr(), "");
        rf.beep((byte) 15);
        return true;
    }

    public void sendApdu() {
        final CpuCard cpu = loop.getCpuCard();
        final byte[] apduData = Convert.hexStrToBytes(Convert.trimHex(String.valueOf(apduDataEdit.getText())));
        if (apduData == null) {
            addLog("发送APDU数据格式错误请正确输入！", "");
            return;
        }
        apduDataEdit.setText(Convert.bytesToHexStr(apduData, apduData.length, " "));
        final byte[] resp = cpu.send(apduData);
        if (resp == null) {
            if (cpuSession)
                addLog("发送失败,COS会话已中断", "请尝试会话终止后重新复位激活！\n");
            else
                addLog("请复位激活COS会话后再试！", "");
            return;
        }
        addLog("COS应答:" + CosErr.getStr(CosErr.cosDataSW(resp)), Convert.bytesToHexStr(resp, resp.length, " ") + "\n");
    }

}