package com.Rokato.jni_demo;

import android.content.Context;
import android.hardware.usb.UsbDevice;
import android.hardware.usb.UsbManager;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.EditText;
import android.widget.Spinner;

import androidx.appcompat.app.AppCompatActivity;

import com.Rokato.jni.Rokato;
import com.Rokato.jni.comm.HidComm;
import com.Rokato.jni.utils.Convert;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

public class HidToolActivity extends AppCompatActivity {
    private static final String TAG = "Rokato_HidTool";
    ArrayAdapter<String> adapter;
    private List<String> devs;
    private HashMap<String, UsbDevice> deviceList;
    private HidComm hid;

    private Spinner devInfoSpinne;
    private Spinner hidModeSpinner;

    private EditText logEdit;
    private EditText interfaceNumEdit;
    private EditText wLengthEdit;
    private EditText baudrateEdit;
    private EditText sendDataEdit;

    private Handler mainHandler;
    private Thread readLoop;
    private boolean flagLoop;
    private final static int millis = 200;
    private long lastUITime = System.currentTimeMillis();
    private long oftenNum = 0;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_hid_tool);

        logEdit = findViewById(R.id.log_edit);
        interfaceNumEdit = findViewById(R.id.interfaceNum_edit);
        wLengthEdit = findViewById(R.id.wLength_edit);
        baudrateEdit = findViewById(R.id.baudrate_edit);
        hidModeSpinner = findViewById(R.id.hidMode_spinner);
        sendDataEdit = findViewById(R.id.sendData_edit);

        Button clearLogBtn = findViewById(R.id.clear_button);
        clearLogBtn.setOnClickListener(v -> logEdit.setText(""));

        devInfoSpinne = findViewById(R.id.devInfo_spinner);
        adapter = new ArrayAdapter<>(this, android.R.layout.simple_spinner_item);
        devInfoSpinne.setAdapter(adapter);

        flushDevs();

        hid = new HidComm(getApplicationContext());
        mainHandler = new Handler(Looper.getMainLooper());
        readLoop = null;
        flagLoop = false;
        addLog("JNI详细信息", Rokato.version() + " " + Rokato.build() + "\n");

        Button flushBtn = findViewById(R.id.flush_button);
        flushBtn.setOnClickListener(v -> flushDevs());

        Button startBtn = findViewById(R.id.start_button);
        startBtn.setOnClickListener(v -> startThread());

        Button stopBtn = findViewById(R.id.stop_button);
        stopBtn.setOnClickListener(v -> {
            stopThread();
            addLog("关闭设备通信完成！", "");
        });

        Button sendBtn = findViewById(R.id.send_button);
        sendBtn.setOnClickListener(v -> sendData());

    }

    private void flushDevs() {
        adapter.clear();
        UsbManager usbManager = (UsbManager) getSystemService(Context.USB_SERVICE);
        deviceList = usbManager.getDeviceList();
        devs = new ArrayList<>();
        if (!deviceList.isEmpty()) {
            for (String key : deviceList.keySet()) {
                UsbDevice device = deviceList.get(key);
                assert device != null;
                String str = Integer.toHexString(device.getVendorId()) + ":" + Integer.toHexString(device.getProductId());
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP)
                    str += "(" + device.getProductName() + "|" + device.getManufacturerName() + ")";
                else
                    str += "(" + device.getDeviceName() + ")";
                devs.add(key);
                adapter.add(str);
            }
        } else {
            adapter.add("查询到的USB设备为空！");
        }
    }

    private void stopThread() {
        if (readLoop != null) {
            while (readLoop.isAlive()) {
                try {
                    flagLoop = false;
                    Thread.sleep(millis);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
        }
        hid.close();
        readLoop = null;
    }

    private void startThread() {
        if (hid.isOpen() && flagLoop) {
            addLog("设备通信正在工作中！", "");
            return;
        }
        if (deviceList.isEmpty()) {
            addLog("当前无可操作的设备！", "");
            return;
        }
        final int wLength = Integer.parseInt(String.valueOf(wLengthEdit.getText()));
        if (wLength < 1 || wLength > 1024) {
            addLog("wLength数据大小范围需1~1024之间且应与端点报文大小一致！", "");
            return;
        }
        if (!hid.open(deviceList.get(devs.get(devInfoSpinne.getSelectedItemPosition())), Integer.parseInt(String.valueOf(interfaceNumEdit.getText())))) {
            addLog("打开设备失败！", "请检查设备或权限是否正常后再试\n");
            return;
        }
        final int hidMode = hidModeSpinner.getSelectedItemPosition();
        if (hidMode == 3) {
            if (!hid.setCH9326(Integer.parseInt(String.valueOf(baudrateEdit.getText())))) {
                hid.close();
                addLog(hidModeSpinner.getSelectedItem().toString() + " 配置串口透传失败！", "");
                return;
            }
        }
        flagLoop = true;
        readLoop = new Thread(() -> {
            byte[] data = new byte[wLength];
            int len;
            while (flagLoop) {
                len = switch (hidMode) {
                    case 0, 3 -> hid.read(data, millis);
                    case 1 -> hid.getFeatureReport(data, millis);
                    case 2 -> hid.getInputReport(data, millis);
                    default -> -99;
                };
                if (len == -7 || len == 0 || (len == -1 && (hidMode == 1 || hidMode == 2)))
                    continue;
                if (len < 0) {
                    addLog("工作终止,读取错误" + len, "");
                    break;
                }
                addLog("收到(" + len + "字节):" + Convert.bytesToHexStr(data, len), "");
            }
            hid.close();
        });
        readLoop.start();
        addLog("启动设备通信完成！", "");
    }

    private void sendData() {
        if (!hid.isOpen()) {
            addLog("请启动设备通信后操作！", "");
            return;
        }
        final byte[] sendData = Convert.hexStrToBytes(Convert.trimHex(String.valueOf(sendDataEdit.getText())));
        if (sendData == null) {
            addLog("发送数据格式错误请正确输入！", "");
            return;
        }
        final int wLength = Integer.parseInt(String.valueOf(wLengthEdit.getText()));
        if (wLength < 1 || wLength > 1024) {
            addLog("wLength数据大小范围需1~1024之间且应与端点报文大小一致！", "");
            return;
        }
        final int hidMode = hidModeSpinner.getSelectedItemPosition();
        byte[] data = sendData;
        if (hidMode == 1) {
            data = new byte[wLength];
            System.arraycopy(sendData, 0, data, 0, Math.min(sendData.length, wLength));
        }
        final int len = switch (hidMode) {
            case 0, 3, 2 -> hid.write(data, data.length);
            case 1 -> hid.sendFeatureReport(data, data.length);
            default -> -99;
        };
        if (len < 0) {
            addLog("发送错误" + len, "");
            return;
        }
        if (len != data.length)
            addLog("发送" + len + "字节成功,失败" + (data.length - len) + "字节", "");
        else
            addLog("发送" + len + "字节成功", "");
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        stopThread();
        hid.free();
        hid = null;
    }

    public void addLog(String tag, String msg) {
        if (Looper.myLooper() == Looper.getMainLooper()) {
            MainActivity.addLog(logEdit, tag, msg);
        } else {
            mainHandler.post(() -> MainActivity.addLog(logEdit, tag, msg));
            if (oftenNum > 2) {
                oftenNum = 0;
                try {
                    Thread.sleep(millis);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
            if (System.currentTimeMillis() - lastUITime < millis) ++oftenNum;
            lastUITime = System.currentTimeMillis();
        }
    }

}