package com.Rokato.jni_demo;

import androidx.appcompat.app.AppCompatActivity;

import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.widget.Button;
import android.widget.EditText;
import android.widget.Spinner;

import com.Rokato.jni.Rokato;
import com.Rokato.jni.rf.IRfReader;
import com.Rokato.jni.rf.LoopReqCard;
import com.Rokato.jni.rf.M1Card;
import com.Rokato.jni.rf.NfcReader;
import com.Rokato.jni.rf.RfReader;
import com.Rokato.jni.utils.Convert;

import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.List;

public class MfToolActivity extends AppCompatActivity {
    private static final String TAG = "Rokato_MfTool";
    private Spinner devTypeSpinner;
    private Spinner keyTypeSpinner;
    private EditText serialAddrEdit;
    private EditText keyEdit;
    private EditText logEdit;
    private EditText sectorEdit;
    private EditText sectorDataEdit;
    private EditText startBlockEdit;
    private EditText blockNumEdit;
    private RfReader rf;
    private NfcReader nfc;
    private LoopReqCard loop;
    private Handler mainHandler;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_mf_tool);

        devTypeSpinner = findViewById(R.id.devTypeSel_spinner);
        keyTypeSpinner = findViewById(R.id.keyTypeSel_spinner);

        serialAddrEdit = findViewById(R.id.serialAddr_edit);
        keyEdit = findViewById(R.id.sectorKey_edit);
        sectorEdit = findViewById(R.id.sector_edit);
        sectorDataEdit = findViewById(R.id.sectorData_edit);
        startBlockEdit = findViewById(R.id.startBlock_edit);
        blockNumEdit = findViewById(R.id.blockNum_edit);

        logEdit = findViewById(R.id.log_edit);

        Button clearLogBtn = findViewById(R.id.clearLog_button);
        clearLogBtn.setOnClickListener(v -> logEdit.setText(""));
        Log.d(TAG, "onCreate: 构造初始化");
        rf = new RfReader(getApplicationContext(), RfReader.DevType_AUTO, RfReader.RfType_ISO14443A);
        nfc = new NfcReader(getApplicationContext());
        loop = new LoopReqCard(100);
        mainHandler = new Handler(Looper.getMainLooper());

        addLog("JNI详细信息", Rokato.version() + " " + Rokato.build() + "\n");

        Button cfgSerialBtn = findViewById(R.id.cfgSerial_button);
        cfgSerialBtn.setOnClickListener(view -> {
            if (rf.serialCfg((byte) devTypeSpinner.getSelectedItemPosition(), String.valueOf(serialAddrEdit.getText())))
                addLog(devTypeSpinner.getSelectedItem().toString() + " 配置串口完成", "");
            else
                addLog(devTypeSpinner.getSelectedItem().toString() + " 不支持配置串口", "");
        });

        nfc.bind(this, tag -> {
            nfc.setNewTag(tag);
            loop.activeLoop();
        });

        Button openDevBtn = findViewById(R.id.openDev_button);
        openDevBtn.setOnClickListener(v -> {
            if (loop.start()) {
                addLog("读卡器工作中,请刷卡！", "");
                return;
            }
            final int pos = devTypeSpinner.getSelectedItemPosition();
            List<IRfReader> devs = new ArrayList<>();
            if (rf.setDevType((byte) pos)) devs.add(rf);
            if (pos == 0 || pos == 7) devs.add(nfc);
            if (loop.start(false, devs, (cardNo, typeBuf, sak) -> {
                final int type = Convert.bytesToUint16(typeBuf, ByteOrder.LITTLE_ENDIAN);
                addLog("发现卡片:" + Convert.getCardNoStr(cardNo), "类型:" + RfReader.typeStr(type) + "  SAK:" + Convert.bytesToHexStr(sak) + "\n");
                if (sak[0] > 0 && (byte) (sak[0] & 0x08) != 0x08) {
                    if (type == 4 || type == 2)
                        addLog("此卡为M1复合卡,本次NFC识别异常无法操作!", "请将卡移开感应区后重新靠近NFC读卡\n");
                    else
                        addLog("此卡可能不是M1卡,无法进行扇区操作!", "");
                }
                rf.beep((byte) 15);
                return true;
            })) {
                if (loop.getRfReader() instanceof RfReader)
                    addLog("打开RF读卡器设备完成", "固件版本:" + rf.getVer() + "\n");
                else if (loop.getRfReader() instanceof NfcReader)
                    addLog("打开NFC读卡器监听完成", "");
                else
                    addLog("未知读卡器启动完成", "");
                return;
            }
            addLog("打开读卡器设备失败！", "请检查读卡器设备或权限是否正常后再试\n");
        });

        Button closeDevBtn = findViewById(R.id.closeDev_button);
        closeDevBtn.setOnClickListener(view -> {
            loop.stop();
            addLog("关闭读卡器设备完成", "");
        });

        Button rfResetBtn = findViewById(R.id.rfReset_button);
        rfResetBtn.setOnClickListener(v -> {
            if (loop.activeReader(this::rfReset)) return;
            addLog("请先打开读卡器设备后操作！", "");
        });

        Button readSectorBtn = findViewById(R.id.readSector_button);
        readSectorBtn.setOnClickListener(v -> {
            if (loop.activeReader(this::readSector)) return;
            addLog("请先打开读卡器设备后操作！", "");
        });

        Button writeSectorBtn = findViewById(R.id.writeSector_button);
        writeSectorBtn.setOnClickListener(v -> {
            if (loop.activeReader(this::writeSector)) return;
            addLog("请先打开读卡器设备后操作！", "");
        });

        Button readAllSectorBtn = findViewById(R.id.readAllSector_button);
        readAllSectorBtn.setOnClickListener(v -> {
            if (loop.activeReader(this::readAllSector)) return;
            addLog("请先打开读卡器设备后操作！", "");
        });

        Button writeAllSectorBtn = findViewById(R.id.writeAllSector_button);
        writeAllSectorBtn.setOnClickListener(v -> {
            if (loop.activeReader(this::writeAllSector)) return;
            addLog("请先打开读卡器设备后操作！", "");
        });
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        loop.stop();
        rf.free();
        rf = null;
        nfc = null;
        Log.d(TAG, "onDestroy: 释放完成");
    }

    public void addLog(String tag, String msg) {
        if (Looper.myLooper() == Looper.getMainLooper()) {
            MainActivity.addLog(logEdit, tag, msg);
        } else {
            mainHandler.post(() -> MainActivity.addLog(logEdit, tag, msg));
        }
    }

    public void rfReset() {
        final M1Card mf = loop.getM1Card();
        byte[] typeBuf = new byte[2];
        byte[] sak = new byte[1];
        if (!mf.reset(false, typeBuf, sak)) {
            addLog("卡片复位激活失败！", "");
            return;
        }
        final int type = Convert.bytesToUint16(typeBuf, ByteOrder.LITTLE_ENDIAN);
        addLog("卡号信息:" + Convert.getCardNoStr(mf.cardNo()) + "  Hex:" + Convert.bytesToHexStr(Convert.uint32ToBytes(mf.cardNo(), ByteOrder.nativeOrder())), "卡类:" + RfReader.typeStr(type) + " SAK:" + Convert.bytesToHexStr(sak) + "\n");
        rf.beep((byte) 15);
    }

    public void readSector() {
        final M1Card mf = loop.getM1Card();
        final byte[] sectorKey = Convert.hexStrToBytes(Convert.trimHex(String.valueOf(keyEdit.getText())));
        if (sectorKey == null) {
            addLog("密钥数据格式错误请正确输入！", "");
            return;
        } else if (sectorKey.length != M1Card.M1_SECTOR_PWD_SIZE) {
            addLog("密钥数据长度错误请正确输入！", "");
            return;
        }
        final byte sector = (byte) Integer.parseInt(String.valueOf(sectorEdit.getText()));
        final byte startBlock = (byte) Integer.parseInt(String.valueOf(startBlockEdit.getText()));
        final byte blockNum = (byte) Integer.parseInt(String.valueOf(blockNumEdit.getText()));
        final boolean keyB = keyTypeSpinner.getSelectedItemPosition() == 1;
        byte[] sectorData = new byte[M1Card.M1_SECTOR_BLOCK_NUM * M1Card.M1_BLOCK_SIZE];
        byte[] blockData = new byte[M1Card.M1_BLOCK_SIZE];
        if (!mf.reset(false, null, null)) {
            addLog("卡片复位激活失败！", "");
            return;
        }
        addLog("操作目标卡号:" + Convert.getCardNoStr(mf.cardNo()), "");
        StringBuilder msg = new StringBuilder();
        final int err = mf.read(true, sector, sectorKey, sectorData, startBlock, blockNum, keyB, true);
        if (err < 1) {
            addLog("扇区" + sector + " 读取失败:" + M1Card.errStr(err), "");
        } else {
            for (byte block = 0; block < err / M1Card.M1_BLOCK_SIZE; ++block) {
                if (msg.length() > 1) msg.append("\n");
                System.arraycopy(sectorData, block * M1Card.M1_BLOCK_SIZE, blockData, 0, blockData.length);
                msg.append(Convert.bytesToHexStr(blockData, blockData.length, " "));
            }
            addLog("扇区" + sector + " 读取" + M1Card.errStr(err), "");
        }
        sectorDataEdit.setText(msg);
        rf.beep((byte) 15);
    }

    public void writeSector() {
        final M1Card mf = loop.getM1Card();
        final byte[] sectorKey = Convert.hexStrToBytes(Convert.trimHex(String.valueOf(keyEdit.getText())));
        if (sectorKey == null) {
            addLog("密钥数据格式错误请正确输入！", "");
            return;
        } else if (sectorKey.length != M1Card.M1_SECTOR_PWD_SIZE) {
            addLog("密钥数据长度错误请正确输入！", "");
            return;
        }
        if (sectorDataEdit.getText().length() < 1)
            sectorDataEdit.setText(getString(R.string.defaultSectorData));
        final byte[] sectorData = Convert.hexStrToBytes(Convert.trimHex(String.valueOf(sectorDataEdit.getText())));
        if (sectorData == null) {
            addLog("写入数据格式错误请正确输入！", "");
            return;
        }
        final byte sector = (byte) Integer.parseInt(String.valueOf(sectorEdit.getText()));
        final byte startBlock = (byte) Integer.parseInt(String.valueOf(startBlockEdit.getText()));
        final byte blockNum = (byte) Integer.parseInt(String.valueOf(blockNumEdit.getText()));
        final boolean keyB = keyTypeSpinner.getSelectedItemPosition() == 1;
        if (!mf.reset(false, null, null)) {
            addLog("卡片复位激活失败！", "");
            return;
        }
        addLog("操作目标卡号:" + Convert.getCardNoStr(mf.cardNo()), "");
        mf.protect(true, true);
        final int err = mf.write(true, sector, sectorKey, sectorData, startBlock, blockNum, keyB);
        mf.protect(false, true);
        if (err < 1)
            addLog("扇区" + sector + " 写入失败:" + M1Card.errStr(err), "");
        else
            addLog("扇区" + sector + " 写入" + M1Card.errStr(err), "");
        rf.beep((byte) 15);
    }

    public void readAllSector() {
        final M1Card mf = loop.getM1Card();
        if (mf == null) {
            addLog("请先打开读卡器设备后操作！", "");
            return;
        }
        final byte[] sectorKey = Convert.hexStrToBytes(Convert.trimHex(String.valueOf(keyEdit.getText())));
        if (sectorKey == null) {
            addLog("密钥数据格式错误请正确输入！", "");
            return;
        } else if (sectorKey.length != M1Card.M1_SECTOR_PWD_SIZE) {
            addLog("密钥数据长度错误请正确输入！", "");
            return;
        }
        final boolean keyB = keyTypeSpinner.getSelectedItemPosition() == 1;
        byte[] sectorData = new byte[M1Card.M1_SECTOR_BLOCK_NUM * M1Card.M1_BLOCK_SIZE];
        byte[] blockData = new byte[M1Card.M1_BLOCK_SIZE];
        int err, ok = 0;
        StringBuilder msg;
        if (!mf.reset(false, null, null)) {
            addLog("卡片复位激活失败！", "");
            return;
        }
        addLog("操作目标卡号:" + Convert.getCardNoStr(mf.cardNo()), "");
        for (byte sector = 0; sector < M1Card.M1_SECTOR_NUM; ++sector) {
            err = mf.read(true, sector, sectorKey, sectorData, (byte) 0, (byte) 0, keyB, true);
            if (err < 1) {
                addLog("扇区" + sector + " 读取失败:" + M1Card.errStr(err), "");
                continue;
            }
            ++ok;
            msg = new StringBuilder();
            for (byte block = 0; block < M1Card.M1_SECTOR_BLOCK_NUM; ++block) {
                System.arraycopy(sectorData, block * M1Card.M1_BLOCK_SIZE, blockData, 0, blockData.length);
                msg.append("块").append(block).append("=").append(Convert.bytesToHexStr(blockData, blockData.length, " ")).append('\n');
            }
            addLog("扇区" + sector + " 读取" + M1Card.errStr(err), msg.toString());
        }
        if (ok == M1Card.M1_SECTOR_NUM)
            addLog("全扇区读取完成", "");
        else
            addLog("全扇区读取结束 成功" + ok + "个扇区，失败" + (M1Card.M1_SECTOR_NUM - ok) + "个扇区！", "");
        rf.beep((byte) 15);
    }

    public void writeAllSector() {
        final M1Card mf = loop.getM1Card();
        final byte[] sectorKey = Convert.hexStrToBytes(Convert.trimHex(String.valueOf(keyEdit.getText())));
        if (sectorKey == null) {
            addLog("密钥数据格式错误请正确输入！", "");
            return;
        } else if (sectorKey.length != M1Card.M1_SECTOR_PWD_SIZE) {
            addLog("密钥数据长度错误请正确输入！", "");
            return;
        }
        if (sectorDataEdit.getText().length() < 1)
            sectorDataEdit.setText(getString(R.string.defaultSectorData));
        final byte[] sectorData = Convert.hexStrToBytes(Convert.trimHex(String.valueOf(sectorDataEdit.getText())));
        if (sectorData == null) {
            addLog("写入数据格式错误请正确输入！", "");
            return;
        }
        final byte startBlock = (byte) Integer.parseInt(String.valueOf(startBlockEdit.getText()));
        final byte blockNum = (byte) Integer.parseInt(String.valueOf(blockNumEdit.getText()));
        final boolean keyB = keyTypeSpinner.getSelectedItemPosition() == 1;
        if (!mf.reset(false, null, null)) {
            addLog("卡片复位激活失败！", "");
            return;
        }
        addLog("操作目标卡号:" + Convert.getCardNoStr(mf.cardNo()), "");
        mf.protect(true, true);
        int err, ok = 0;
        for (byte sector = 0; sector < M1Card.M1_SECTOR_NUM; ++sector) {
            err = mf.write(true, sector, sectorKey, sectorData, startBlock, blockNum, keyB);
            if (err == sectorData.length || (sector == 0 && startBlock < 1 && err == sectorData.length - M1Card.M1_BLOCK_SIZE)) {
                ++ok;
                addLog("扇区" + sector + " 写入" + M1Card.errStr(err), "");
            } else {
                addLog("扇区" + sector + " 写入失败:" + M1Card.errStr(err), "");
            }
        }
        mf.protect(false, true);
        if (ok == M1Card.M1_SECTOR_NUM)
            addLog("全扇区写入完成 ", "");
        else
            addLog("全扇区写入结束 成功" + ok + "个扇区，失败" + (M1Card.M1_SECTOR_NUM - ok) + "个扇区！", "");
        rf.beep((byte) 15);
    }
}