package com.Rokato.jni_demo;

import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.view.View;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.EditText;
import android.widget.Spinner;

import androidx.appcompat.app.AppCompatActivity;

import com.Rokato.jni.Rokato;
import com.Rokato.jni.comm.SerialPort;
import com.Rokato.jni.utils.Convert;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class SerialToolActivity extends AppCompatActivity {
    private static final String TAG = "Rokato_SerialTool";
    private SerialPort serial;
    ArrayAdapter<String> adapter;

    private EditText serialAddrEdit;
    private EditText baudrateEdit;
    private EditText sendDataEdit;
    private Spinner flowCtlSpinner;
    private Spinner paritySpinner;
    private Spinner databitsSpinner;
    private Spinner stopbitsSpinner;

    private EditText logEdit;
    private Handler mainHandler;
    private Thread readLoop;
    private boolean flagLoop;
    private final static int millis = 200;
    private final static int interByteMs = 1;
    private long lastUITime = System.currentTimeMillis();
    private long oftenNum = 0;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_serial_tool);

        serialAddrEdit = findViewById(R.id.serialAddr_edit);
        logEdit = findViewById(R.id.log_edit);
        baudrateEdit = findViewById(R.id.baudrate_edit);
        sendDataEdit = findViewById(R.id.sendData_edit);

        flowCtlSpinner = findViewById(R.id.flowCtl_spinner);
        paritySpinner = findViewById(R.id.parity_spinner);
        databitsSpinner = findViewById(R.id.databits_spinner);
        stopbitsSpinner = findViewById(R.id.stopbits_spinner);

        databitsSpinner.setSelection(4);

        Button clearLogBtn = findViewById(R.id.clear_button);
        clearLogBtn.setOnClickListener(v -> logEdit.setText(""));

        Spinner serialInfoSpinne = findViewById(R.id.serialInfo_spinner);
        adapter = new ArrayAdapter<>(this, android.R.layout.simple_spinner_item);
        serialInfoSpinne.setAdapter(adapter);

        flushPorts();

        serialInfoSpinne.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> parent, View view, int position, long id) {
                String str = parent.getSelectedItem().toString();
                final int pos = str.indexOf("(");
                if (pos < 1) return;
                serialAddrEdit.setText(str.substring(0, pos));
            }

            @Override
            public void onNothingSelected(AdapterView<?> parent) {

            }
        });

        serial = new SerialPort(9600, millis);
        mainHandler = new Handler(Looper.getMainLooper());
        readLoop = null;
        flagLoop = false;
        addLog("JNI详细信息", Rokato.version() + " " + Rokato.build() + "\n");

        Button flushBtn = findViewById(R.id.flush_button);
        flushBtn.setOnClickListener(v -> flushPorts());

        Button startBtn = findViewById(R.id.start_button);
        startBtn.setOnClickListener(v -> startThread());

        Button stopBtn = findViewById(R.id.stop_button);
        stopBtn.setOnClickListener(v -> {
            stopThread();
            addLog("关闭串口通信完成！", "");
        });

        Button sendBtn = findViewById(R.id.send_button);
        sendBtn.setOnClickListener(v -> sendData());
    }

    private void flushPorts() {
        adapter.clear();
        String json = SerialPort.enumPorts();
        if (json != null) {
            try {
                JSONArray jsonObj = new JSONArray(json);
                for (int i = 0; i < jsonObj.length(); ++i) {
                    JSONObject portObj = jsonObj.getJSONObject(i);
                    adapter.add(portObj.getString("addr") + "(" + portObj.getString("desc") + "|" + portObj.getString("devId") + ")");
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }
        } else {
            adapter.add("查询到的端口信息为空！");
        }
    }

    private void stopThread() {
        if (readLoop != null) {
            while (readLoop.isAlive()) {
                try {
                    flagLoop = false;
                    Thread.sleep(millis);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
        }
        serial.close();
        readLoop = null;
    }

    private void startThread() {
        if (serial.isOpen() && flagLoop) {
            addLog("串口通信正在工作中！", "");
            return;
        }
        serial.free();
        serial = new SerialPort(
                Integer.parseInt(String.valueOf(baudrateEdit.getText())),
                millis, interByteMs,
                (byte) paritySpinner.getSelectedItemPosition(),
                (byte) (databitsSpinner.getSelectedItemPosition() + SerialPort.Databits_4),
                (byte) (stopbitsSpinner.getSelectedItemPosition() + SerialPort.Stopbits_one),
                (byte) flowCtlSpinner.getSelectedItemPosition()
        );
        if (!serial.setAddr(serialAddrEdit.getText().toString())) {
            addLog("配置串口地址失败！", "");
            return;
        }
        if (!serial.open()) {
            addLog("打开串口失败！", "请检查端口地址是否可用后再试\n");
            return;
        }

        flagLoop = true;
        readLoop = new Thread(() -> {
            byte[] data = new byte[1024];
            int len;
            while (flagLoop) {
                len = serial.read(data);
                if (len == 0) continue;
                if (len < 0) {
                    addLog("工作终止,读取错误" + len, "");
                    break;
                }
                addLog("收到(" + len + "字节):" + Convert.bytesToHexStr(data, len), "");
            }
            serial.close();
        });
        readLoop.start();
        addLog("启动串口通信完成！", "");
    }

    private void sendData() {
        if (!serial.isOpen()) {
            addLog("请启动串口通信后操作！", "");
            return;
        }
        final byte[] data = Convert.hexStrToBytes(Convert.trimHex(String.valueOf(sendDataEdit.getText())));
        if (data == null) {
            addLog("发送数据格式错误请正确输入！", "");
            return;
        }

        final int len = serial.write(data, data.length);
        if (len < 0) {
            addLog("发送错误" + len, "");
            return;
        }
        if (data.length == len)
            addLog("发送" + data.length + "字节成功", "");
        else
            addLog("发送" + len + "字节成功,失败" + (data.length - len) + "字节", "");
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        stopThread();
        serial.free();
        serial = null;
    }

    public void addLog(String tag, String msg) {
        if (Looper.myLooper() == Looper.getMainLooper()) {
            MainActivity.addLog(logEdit, tag, msg);
        } else {
            mainHandler.post(() -> MainActivity.addLog(logEdit, tag, msg));
            if (oftenNum > 2) {
                oftenNum = 0;
                try {
                    Thread.sleep(millis);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
            if (System.currentTimeMillis() - lastUITime < millis) ++oftenNum;
            lastUITime = System.currentTimeMillis();
        }
    }
}