package com.Rokato.jni_demo;

import android.annotation.SuppressLint;
import android.content.Context;
import android.hardware.usb.UsbDevice;
import android.hardware.usb.UsbManager;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.view.View;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.EditText;
import android.widget.Spinner;

import androidx.appcompat.app.AppCompatActivity;

import com.Rokato.jni.Rokato;
import com.Rokato.jni.comm.UsbComm;
import com.Rokato.jni.utils.Convert;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

public class UsbToolActivity extends AppCompatActivity {
    private static final String TAG = "Rokato_UsbTool";

    ArrayAdapter<String> adapter;
    private List<String> devs;
    private HashMap<String, UsbDevice> deviceList;
    private UsbComm usb;

    private Spinner devInfoSpinne;
    private Spinner usbModeSpinner;

    private EditText logEdit;
    private EditText configNumEdit;
    private EditText interfaceNumEdit;
    private EditText epAddrEdit;
    private EditText requestTypeEdit;
    private EditText bRequestEdit;
    private EditText wValueEdit;
    private EditText wIndexEdit;
    private EditText wLengthEdit;
    private EditText baudrateEdit;
    private EditText sendDataEdit;

    private Handler mainHandler;
    private Thread readLoop;
    private boolean flagLoop;
    private final static int millis = 200;
    private final static int interByteMs = 50;
    private long lastUITime = System.currentTimeMillis();
    private long oftenNum = 0;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_usb_tool);

        logEdit = findViewById(R.id.log_edit);
        configNumEdit = findViewById(R.id.configNum_edit);
        interfaceNumEdit = findViewById(R.id.interfaceNum_edit);
        epAddrEdit = findViewById(R.id.epAddr_edit);
        requestTypeEdit = findViewById(R.id.requestType_edit);
        bRequestEdit = findViewById(R.id.bRequest_edit);
        wValueEdit = findViewById(R.id.wValue_edit);
        wIndexEdit = findViewById(R.id.wIndex_edit);
        wLengthEdit = findViewById(R.id.wLength_edit);
        baudrateEdit = findViewById(R.id.baudrate_edit);
        usbModeSpinner = findViewById(R.id.usbMode_spinner);
        sendDataEdit = findViewById(R.id.sendData_edit);

        Button clearLogBtn = findViewById(R.id.clear_button);
        clearLogBtn.setOnClickListener(v -> logEdit.setText(""));

        devInfoSpinne = findViewById(R.id.devInfo_spinner);
        adapter = new ArrayAdapter<>(this, android.R.layout.simple_spinner_item);
        devInfoSpinne.setAdapter(adapter);

        flushDevs();

        usb = new UsbComm(getApplicationContext());
        mainHandler = new Handler(Looper.getMainLooper());
        readLoop = null;
        flagLoop = false;
        addLog("JNI详细信息", Rokato.version() + " " + Rokato.build() + "\n");

        usbModeSpinner.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> parent, View view, int position, long id) {
                activeDefaultCfg(position, usb.isOpen() && flagLoop);
            }

            @Override
            public void onNothingSelected(AdapterView<?> parent) {

            }
        });

        Button flushBtn = findViewById(R.id.flush_button);
        flushBtn.setOnClickListener(v -> flushDevs());

        Button startBtn = findViewById(R.id.start_button);
        startBtn.setOnClickListener(v -> startThread());

        Button stopBtn = findViewById(R.id.stop_button);
        stopBtn.setOnClickListener(v -> {
            stopThread();
            addLog("关闭设备通信完成！", "");
        });

        Button sendBtn = findViewById(R.id.send_button);
        sendBtn.setOnClickListener(v -> sendData());

        Button restartDevBtn = findViewById(R.id.restartDev_button);
        restartDevBtn.setOnClickListener(v -> {
            if (!usb.isOpen()) {
                addLog("请启动设备通信后操作！", "");
                return;
            }
            final int err = usb.resetDevice();
            if (err == 0) {
                stopThread();
                flushDevs();
                addLog("重启设备完成！", "");
            } else {
                addLog("重启设备错误:" + err, "");
            }
        });

        Button clearHaltBtn = findViewById(R.id.clearHalt_button);
        clearHaltBtn.setOnClickListener(v -> {
            if (!usb.isOpen()) {
                addLog("请启动设备通信后操作！", "");
                return;
            }
            final int epAddr = Integer.parseInt(String.valueOf(epAddrEdit.getText()));
            if (epAddr < 0 || epAddr > 15) {
                addLog("端点地址范围必须在0~15之间！", "");
                return;
            }
            final int err = usb.clearHalt((byte) epAddr);
            if (err == 0)
                addLog("清除端点Halt完成！", "");
            else
                addLog("清除端点Halt错误:" + err, "");
        });
    }

    @SuppressLint("SetTextI18n")
    private void activeDefaultCfg(int usbMode, boolean hasSendEP) {
        switch (usbMode) {
            case 3:
                configNumEdit.setText("0");
                interfaceNumEdit.setText("0");
                epAddrEdit.setText("2");
                break;

            case 4:
                configNumEdit.setText("0");
                interfaceNumEdit.setText("0");
                epAddrEdit.setText(hasSendEP ? "2" : "3");
                break;

            case 5:
                configNumEdit.setText("0");
                interfaceNumEdit.setText("0");
                epAddrEdit.setText(hasSendEP ? "2" : "1");
                break;

            case 6:
                configNumEdit.setText("0");
                interfaceNumEdit.setText("0");
                epAddrEdit.setText("4");
                wLengthEdit.setText("64");
                break;

            default:
                break;
        }
    }

    private void flushDevs() {
        adapter.clear();
        UsbManager usbManager = (UsbManager) getSystemService(Context.USB_SERVICE);
        deviceList = usbManager.getDeviceList();
        devs = new ArrayList<>();
        if (!deviceList.isEmpty()) {
            for (String key : deviceList.keySet()) {
                UsbDevice device = deviceList.get(key);
                assert device != null;
                String str = Integer.toHexString(device.getVendorId()) + ":" + Integer.toHexString(device.getProductId());
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP)
                    str += "(" + device.getProductName() + "|" + device.getManufacturerName() + ")";
                else
                    str += "(" + device.getDeviceName() + ")";
                devs.add(key);
                adapter.add(str);
            }
        } else {
            adapter.add("查询到的USB设备为空！");
        }
    }

    private void stopThread() {
        if (readLoop != null) {
            while (readLoop.isAlive()) {
                try {
                    flagLoop = false;
                    Thread.sleep(millis);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
        }
        usb.close();
        readLoop = null;
        activeDefaultCfg(usbModeSpinner.getSelectedItemPosition(), false);
    }


    private void startThread() {
        if (usb.isOpen() && flagLoop) {
            addLog("设备通信正在工作中！", "");
            return;
        }
        if (deviceList.isEmpty()) {
            addLog("当前无可操作的设备！", "");
            return;
        }
        final int usbMode = usbModeSpinner.getSelectedItemPosition();
        final int epAddr = Integer.parseInt(String.valueOf(epAddrEdit.getText()));
        if (epAddr < 0 || epAddr > 15) {
            addLog("端点地址范围必须在0~15之间！", "");
            return;
        }
        final int wLength = Integer.parseInt(String.valueOf(wLengthEdit.getText()));
        if (wLength < 0 || wLength > 1024) {
            addLog("wLength数据大小范围需0~1024之间且应与端点报文大小一致！", "");
            return;
        }
        if (!usb.open(deviceList.get(devs.get(devInfoSpinne.getSelectedItemPosition())), -1)) {
            addLog("打开设备失败！", "请检查设备或权限是否正常后再试\n");
            return;
        }
        int err;
        final int configId = Integer.parseInt(String.valueOf(configNumEdit.getText()));
        if (configId > 0) {
            err = usb.setConfig(configId);
            if (err != 0) {
                addLog("激活配置号错误:" + err, "");
                usb.close();
                return;
            }
        }
        err = usb.claimInterface(Integer.parseInt(String.valueOf(interfaceNumEdit.getText())));
        if (err != 0) {
            addLog("激活接口号错误:" + err, "");
            usb.close();
            return;
        }

        if (usbMode == 3)
            err = usb.setCH340(Integer.parseInt(String.valueOf(baudrateEdit.getText())), interByteMs);
        else if (usbMode == 4)
            err = usb.setUSBSerialControllerD(Integer.parseInt(String.valueOf(baudrateEdit.getText())), interByteMs);
        else if (usbMode == 5)
            err = usb.setFT232Uart(Integer.parseInt(String.valueOf(baudrateEdit.getText())), interByteMs, 2);
        else if (usbMode == 6)
            err = usb.setCH9329Cfg(0, 0, (short) 3, (short) 2, Integer.parseInt(String.valueOf(baudrateEdit.getText())), (short) 0, interByteMs);
        else if (usbMode == 2) {
            flagLoop = true;
            addLog("启动设备通信完成！", "");
            return;
        }
        if (err != 0) {
            usb.close();
            addLog(usbModeSpinner.getSelectedItem().toString() + " 配置串口透传错误:" + err, "");
            return;
        }
        if (usbMode == 6) addLog("注意CH9329更改固件配置需重新插拔生效！", "");

        activeDefaultCfg(usbMode, true);
        flagLoop = true;
        readLoop = new Thread(() -> {
            final short ep = (short) (epAddr | 0x80);
            byte[] data = new byte[wLength];
            int len;
            while (flagLoop) {
                len = switch (usbMode) {
                    case 0, 3, 4, 5 -> usb.bulkTransfer(ep, data, data.length, millis);
                    case 1, 6 -> usb.interruptTransfer(ep, data, data.length, millis);
                    default -> -99;
                };
                if (len == -7 || len == 0) continue;
                if (len < 0) {
                    addLog("工作终止,读取错误" + len, "");
                    break;
                }
                if (usbMode == 6) len = 1 + data[0];
                addLog("收到(" + len + "字节):" + Convert.bytesToHexStr(data, len), "");
            }
            usb.close();
        });
        readLoop.start();
        addLog("启动设备通信完成！", "");
    }

    private void sendData() {
        if (!usb.isOpen()) {
            addLog("请启动设备通信后操作！", "");
            return;
        }
        final int usbMode = usbModeSpinner.getSelectedItemPosition();
        final short epAddr = (short) Integer.parseInt(String.valueOf(epAddrEdit.getText()));
        if (epAddr < 0 || epAddr > 15) {
            addLog("端点地址范围必须在0~15之间！", "");
            return;
        }
        final int bmRequestType = Integer.parseInt(String.valueOf(requestTypeEdit.getText()), 16);
        if (bmRequestType < 0 || bmRequestType > 255) {
            addLog("bmRequestType范围需00~FF之间且应符合USB协议规范！", "");
            return;
        }
        final int bRequest = Integer.parseInt(String.valueOf(bRequestEdit.getText()), 16);
        if (bRequest < 0 || bRequest > 255) {
            addLog("bRequest范围需00~FF之间且应符合USB协议规范！", "");
            return;
        }
        final int wValue = Integer.parseInt(String.valueOf(wValueEdit.getText()), 16);
        if (wValue < 0 || wValue > 65535) {
            addLog("wValue范围需0000~FFFF之间且应符合USB协议规范！", "");
            return;
        }
        final int wIndex = Integer.parseInt(String.valueOf(wIndexEdit.getText()), 16);
        if (wIndex < 0 || wIndex > 65535) {
            addLog("wIndex范围需0000~FFFF之间且应符合USB协议规范！", "");
            return;
        }
        final int wLength = Integer.parseInt(String.valueOf(wLengthEdit.getText()));
        if (wLength < 0 || wLength > 1024) {
            addLog("wLength数据大小范围需0~1024之间且应与端点报文大小一致！", "");
            return;
        }

        final byte[] sendData = Convert.hexStrToBytes(Convert.trimHex(String.valueOf(sendDataEdit.getText())));
        if (usbMode != 2 && sendData == null) {
            addLog("发送数据格式错误请正确输入！", "");
            return;
        }
        byte[] data = sendData;
        if (usbMode == 2 && wLength < 1) {
            data = null;
        } else if (usbMode == 2) {
            if (sendData.length > wLength) {
                addLog("控制传输数据大小不可超过wLength报文大小！", "");
                return;
            }
            data = new byte[wLength];
            System.arraycopy(sendData, 0, data, 0, sendData.length);
        } else if (usbMode == 6) {
            if (sendData.length > wLength) {
                addLog("USB-HID传输数据大小不可超过wLength报文大小！", "");
                return;
            }
            data = new byte[wLength + 1];
            data[0] = (byte) sendData.length;
            System.arraycopy(sendData, 0, data, 1, data[0]);
        }
        final int len = switch (usbMode) {
            case 0, 3, 4, 5 -> usb.bulkTransfer(epAddr, data, data.length, millis);
            case 1, 6 -> usb.interruptTransfer(epAddr, data, data.length, millis);
            case 2 ->
                    usb.controlTransfer((byte) bmRequestType, (byte) bRequest, (short) wValue, (short) wIndex, data, millis);
            default -> -99;
        };
        if (len < 0) {
            addLog("发送错误" + len, "");
            return;
        }
        if (data == null) {
            addLog("发送控制报文完成", "");
            return;
        }
        if (data.length == len || usbMode == 2)
            addLog("发送" + data.length + "字节成功", usbMode == 2 ? "响应(" + len + "字节):" + Convert.bytesToHexStr(data, len) + "\n" : "");
        else
            addLog("发送" + len + "字节成功,失败" + (data.length - len) + "字节", "");
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        stopThread();
        usb.free();
        usb = null;
    }

    public void addLog(String tag, String msg) {
        if (Looper.myLooper() == Looper.getMainLooper()) {
            MainActivity.addLog(logEdit, tag, msg);
        } else {
            mainHandler.post(() -> MainActivity.addLog(logEdit, tag, msg));
            if (oftenNum > 2) {
                oftenNum = 0;
                try {
                    Thread.sleep(millis);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
            if (System.currentTimeMillis() - lastUITime < millis) ++oftenNum;
            lastUITime = System.currentTimeMillis();
        }
    }

}